
const double tiny = 1.0e-16;
const double pi = 3.14159265358979;

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <time.h>
#include <omp.h>

void tridag(double *a,double *b,double *c,double *r,double *gam,double *u,long N)
{

  long j;
  double bet;

  if (b[0] == 0.0) printf("Error 1 in tridag");
  u[0] = r[0]/(bet = b[0]);
  for (j=1;j<N;j++) {
    gam[j] = c[j-1]/(bet+tiny);
    bet = b[j] - a[j]*gam[j];
    /*if (bet == 0.0) nrerror("Error 2 in tridag");*/
    u[j] = (r[j]-a[j]*u[j-1])/(bet+tiny);
  }
  for (j=(N-2);j>=0;j--) u[j] -= gam[j+1]*u[j+1];

}

void steady_state(int N,int Ne,double conBR,double conICE,double conWAT,double w,double Tliq,double Tsol,double *po,double qb,double T0,double Tb,double *dz,double *con,double *fracWAT,double **Tcoef,double *T,double *Te,double *dTdz)
{

  int i,j;
  double conF,conU;  

  /*initiate*/
  for (i=0;i<5;i++) for (j=0;j<N;j++) Tcoef[i][j] = 0.0;

  /*loop elements*/
  for (i=0;i<Ne;i++) {

    /*Thermal conduction*/
    conF =  pow(conICE,po[i])*pow(conBR,1-po[i]);
    conU =  pow(conWAT,po[i])*pow(conBR,1-po[i]);
    if (T[i]>Tliq) { /*Unfrozen phase*/
      con[i] = conU;
    }
    else if (T[i]<Tsol) { /*Frozen phase*/
      con[i] = conF;
    }
    else { /* Freezing/thawing phase */
      con[i] = pow(conF,1.0-fracWAT[i])*pow(conU,fracWAT[i]);
    }

    Tcoef[1][i] += con[i]/dz[i];
    Tcoef[2][i] -= con[i]/dz[i];
    Tcoef[0][i+1] -= con[i]/dz[i];
    Tcoef[1][i+1] += con[i]/dz[i];

  }

  
  /*essential boundary condition at top*/
  Tcoef[3][1] -= T0*Tcoef[0][1];
  Tcoef[3][0] = T0;
  Tcoef[0][1] = 0.0;
  Tcoef[2][0] = 0.0;
  Tcoef[1][0] = 1.0;
  
 /*essential boundary condition at base*/
  /*Tcoef[3][N-2] -= Tb*Tcoef[2][N-2];
  Tcoef[3][N-1] = Tb;
  Tcoef[2][N-2] = 0.0;
  Tcoef[0][N-1] = 0.0;
  Tcoef[1][N-1] = 1.0;*/
  
  /*natural boundary conditon at base*/
  Tcoef[3][N-1] += qb;
  
  /*solve the tridiagonal system*/
  tridag(Tcoef[0],Tcoef[1],Tcoef[2],Tcoef[3],Tcoef[4],T,N);

  /*element temperature and gradient*/
  for (i=0;i<Ne;i++) {
    Te[i] = .5*(T[i]+T[i+1]);
    dTdz[i] = (T[i+1]-T[i])/dz[i];
  }

}


void update_transient(int N,int Ne,double conBR,double conICE,double conWAT,double denBR,double denICE,double denWAT,double cpBR,double cpICE,double cpWAT,double L,double w,double Tliq,double Tsol,double *po,double dt,double qb,double Ts,double Tb,double *dz,double *fracWAT,double *dfracWAT,double *con,double *cv,double *denBulk,double **Tcoef,double *T,double *Tt,double *Te,double *dTdz,int *cnn)
{

  int i,j;
  int cn;
  double spd = 24.0*3600.0; /*seconds per day*/
  double conF, conU, fac, cv_f, cv_u;
  double Tte,Te_old,dTe;
  double dM1,dM2,dM3,dM4,dM;

  /*initiate arays*/
  for (i=0;i<5;i++) for (j=0;j<N;j++) Tcoef[i][j] = 0.0;

  /*loop elements for assembling of system*/ 
  for (i=0;i<Ne;i++) {

    /*Thermal conduction & Specific heat*/
    conF =  pow(conICE,po[i])*pow(conBR,1.0-po[i]);
    conU =  pow(conWAT,po[i])*pow(conBR,1.0-po[i]);
    denBulk[i] = (1.0-po[i])*denBR+po[i]*fracWAT[i]*denWAT+po[i]*(1.0-fracWAT[i])*denICE;
    cv_f = (1.0-po[i])*denBR*cpBR+po[i]*denICE*cpICE;
    cv_u = (1.0-po[i])*denBR*cpBR+po[i]*denWAT*cpWAT;
    cnn[i] = 0;

    if ((T[i]>Tliq)&&(fracWAT[i] >= 1.0)) { /*Unfrozen*/
        con[i] = conU;
        cv[i] = cv_u;
     }
    else if ((T[i]<Tsol)&&(fracWAT[i] <= 0.0)) { /*Frozen*/
        con[i] = conF;
        cv[i] = cv_f;
     }
    else { /*Phase-change*/
      /*con[i] = conF*(1.0-fracWAT[i])+conU*fracWAT[i];*/
      con[i] = pow(conF,1.0-fracWAT[i])*pow(conU,fracWAT[i]);
      cv[i] = cv_f+L*(0.95*po[i]*denWAT)/(Tliq-Tsol);
      cnn[i] = 1;
     }

    fac = cv[i]/(dt*spd);
   
    /*Crank-Nicolson*/
    Tcoef[1][i] += 0.5*con[i]/dz[i]+fac*dz[i]/3.0;
    Tcoef[2][i] += -0.5*con[i]/dz[i]+fac*dz[i]/6.0;
    Tcoef[0][i+1] += -0.5*con[i]/dz[i]+fac*dz[i]/6.0;
    Tcoef[1][i+1] += 0.5*con[i]/dz[i]+fac*dz[i]/3.0;
    Tcoef[3][i] += fac*(T[i]/3.0+T[i+1]/6.0)*dz[i]-0.5*con[i]/dz[i]*(T[i]-T[i+1]);
    Tcoef[3][i+1] += fac*(T[i]/6.0+T[i+1]/3.0)*dz[i]-0.5*con[i]/dz[i]*(-T[i]+T[i+1]);
  }
   
  
  /*essential boundary condition at top*/
  Tcoef[3][1] -= Ts*Tcoef[0][1];
  Tcoef[3][0] = Ts;
  Tcoef[0][1] = 0.0;
  Tcoef[2][0] = 0.0;
  Tcoef[1][0] = 1.0;
  
  /*essential boundary condition at base*/
  /*Tcoef[3][N-2] -= Tb*Tcoef[2][N-2];
  Tcoef[3][N-1] = Tb;
  Tcoef[2][N-2] = 0.0;
  Tcoef[0][N-1] = 0.0;
  Tcoef[1][N-1] = 1.0;*/
 
  /*natural boundary conditon at base*/
  Tcoef[3][N-1] += qb;
    
  /*solve the tridiagonal system*/
  tridag(Tcoef[0],Tcoef[1],Tcoef[2],Tcoef[3],Tcoef[4],Tt,N);

  /*loop elements for update on phase change*/
  for (i=0;i<Ne;i++) {
    if (cnn[i] > 0) {
      Te_old = .5*(T[i]+T[i+1]);
      Tte = .5*(Tt[i]+Tt[i+1]);
      dTe = Tte - Te_old;
      dM = dTe/(Tliq-Tsol);
    }
    else dM = 0.0;
    /*if ((Te[i] > Tliq)&&(fracWAT[i] < 1.0)) dM += 0.001*(Te[i]-Tliq);
      if ((Te[i] < Tsol)&&(fracWAT[i] > 0.0)) dM += 0.001*(Te[i]-Tsol);*/
    if (dM > 0.0) {
      if (fracWAT[i] >= 1.0) dM = 0.0;
      else if ((fracWAT[i] + dM) > 1.0) dM = 1.0 - fracWAT[i];
    }
    if (dM < 0.0) {
      if (fracWAT[i] <= 0.0) dM = 0.0;
      else if ((fracWAT[i] + dM) < 0.0) dM = -fracWAT[i];
    }
    dfracWAT[i] = dM;
    fracWAT[i] += dfracWAT[i];
  }
    
  /*accept solution*/
  for (i=0;i<N;i++) T[i] = Tt[i];

  /*element temperature and gradient*/
  for (i=0;i<Ne;i++) {
    dTdz[i] = (T[i+1]-T[i])/dz[i];
    Te[i] = .5*(T[i]+T[i+1]);
  }

}


void frost_cracking(int Ne,double dt,int *sed,double *dz,double *ze,double *po,double *dTdz,double *Te,double *Cr)
{

  int i,nnr;
  double dL,Vw;
  double dLc = 4.0;
  double dLs = 1.0;
  double dLb = 2.0;
  double dLsc = 2.0;
  double Vcrit = 0.04;

  /*loop elements*/
  for (i=0;i<Ne;i++) {
    
    /*if element is bedrock in frost cracking window*/
    if ((Te[i] > -8.0)&&(Te[i] < -3.0)&&(sed[i] == 0)) {
      
      /*initialize*/
      dL = 0.0; Vw = 0.0; nnr = i;

      /*if water comes from above*/
      if (dTdz[i] < 0.0) {
	while ((nnr >= 0)&&(dTdz[nnr] < 0.0)) {
	  if (sed[nnr] == 1) {
	    if (Te[nnr] > 0.0) {
	      dL += dLs*dz[nnr];
	      Vw += dz[nnr]*po[nnr]*exp(-dL);
	    }
	    else dL += dLsc*dz[nnr];
	  }
	  else if (Te[nnr] <= 0.0) {
	    dL += dLc*dz[nnr];
	  }
	  else {
	    dL += dLb*dz[nnr];
	    Vw += dz[nnr]*po[nnr]*exp(-dL);
	  }
	  nnr -= 1;
	}
      }

      /*else if water comes from below*/
      else if (dTdz[i] > 0.0) {
	while ((nnr < Ne)&&(dTdz[nnr] > 0.0)) {
	  if (sed[nnr] == 1) {
	    if (Te[nnr] > 0.0) {
	      dL += dLs*dz[nnr];
	      Vw += dz[nnr]*po[nnr]*exp(-dL);
	    }
	    else dL += dLsc*dz[nnr];
	  }
	  else if (Te[nnr] <= 0.0) {
	    dL += dLc*dz[nnr];
	  }
	  else {
	    dL += dLb*dz[nnr];
	    Vw += dz[nnr]*po[nnr]*exp(-dL);
	  }
	  nnr += 1;
	}
      }

      /*limit water content*/
      if (Vw > Vcrit) Vw = Vcrit;

      /*add contribution to frost cracking*/
      Cr[i] += fabs(dTdz[i])*Vw*dt;

    }/*if*/

  }/*i*/

}/*void*/

void get_transport(int Ne,double beta,double time,double dt,int *sed,double *dfracWAT,double *ze,double *dz,double *Te,double *dwv,double *dwvs)
{

  int i;
  double mindw = 0.0*1.0e-2*dt;

  /*loop elements*/
  for (i=0;i<Ne;i++) {

    if ((sed[i] == 1)&&(fabs(dfracWAT[i]) > mindw)) {
      dwv[i] += fabs(dfracWAT[i]);
      dwvs[i] += dfracWAT[i];
    }

  }

}

void annual_integration(int N,int Ne,int Nt,double *Temp,double conBR,double conICE,double conWAT,double denBR,double denICE,double denWAT,double cpBR,double cpICE,double cpWAT,double L,double w, double Tliq, double Tsol,double dTa,double qb,double T0,double po_s,double po_b,double beta,double totaltime,double dt,double Hs,int *sed,double *z,double *ze,double *dz,double *po,double *T,double *Tt,double *Te,double *dTdz,double **Tcoef,double *Cr,double *dwv,double *dwvs,double *fracWAT,double *dfracWAT,double *con,double *cv,double *denBulk,double *Tr,double *Ci,int *cnn,int makeoutput)
{

  int i,tt,yy;
  int Ntotal = (int)(totaltime*Nt);
  double Ts;
  FILE *fp;

  double time = 0.0;

  /*initalize transport*/
  (*Tr) = 0.0;

  /*initialize element arrays*/
  for (i=0;i<Ne;i++) {
    Cr[i] = 0.0;
    dwv[i] = 0.0;
    dwvs[i] = 0.0;
    fracWAT[i] = 0.0;
    if (ze[i] < Hs) {
      sed[i] = 1;
      po[i] = po_s;
    }
    else {
      sed[i] = 0;
      po[i] = po_b;
    }
  }

  /*get steady state solution*/
  steady_state(N,Ne,conBR,conICE,conWAT,w,Tliq,Tsol,po,qb,T0,T0,dz,con,fracWAT,Tcoef,T,Te,dTdz);

  /*initiate freezing array*/
  for (i=0;i<Ne;i++) {
    if (Te[i] < Tsol) {
      fracWAT[i] = 0.0;
    }
    else if (Te[i] > Tliq) {
      fracWAT[i] = 1.0;
    }
    else {
      fracWAT[i] = (Te[i]-Tsol)/(Tliq-Tsol);
    }
  }

  /*if output required*/
  if (makeoutput > 0) {

    /*create output file*/
    if ((fp = fopen("output.dat","wb")) == NULL) {
      printf("could not open file for writing\n");
      exit(1);
    }
    else {
      fwrite(&N,sizeof(int),1,fp);
      fwrite(&Nt,sizeof(int),1,fp);
      fwrite(&Hs,sizeof(double),1,fp);
      fwrite(&Tsol,sizeof(double),1,fp);
      for (i=0;i<N;i++) fwrite(&z[i],sizeof(double),1,fp);
      for (i=0;i<N;i++) fwrite(&T[i],sizeof(double),1,fp);
    }
    fclose(fp);
  
  }

  /*loop years*/
  for (yy=0;yy<(int)totaltime;yy++) {

    /*begin the transient stage*/
    for (tt=0;tt<Nt;tt++) {

      time += dt;
      
      /*update surface temperature*/
      Ts = T0 + Temp[tt] + dTa*sin(2*pi*time/365.0);
      
      /*update transient solution*/
      update_transient(N,Ne,conBR,conICE,conWAT,denBR,denICE,denWAT,cpBR,cpICE,cpWAT,L,w,Tliq,Tsol,po,dt,qb,Ts,T0,dz,fracWAT,dfracWAT,con,cv,denBulk,Tcoef,T,Tt,Te,dTdz,cnn);
      
      /*if in third year*/
      if (time > ((int)totaltime-1)*365.0) {
	
	/*add to frost cracking intensity*/
	frost_cracking(Ne,dt,sed,dz,ze,po,dTdz,Te,Cr);
	
	/*add to transport capacity*/
	get_transport(Ne,beta,time,dt,sed,dfracWAT,ze,dz,Te,dwv,dwvs);
	
      }/*if*/


      /*if output required*/
      if ((makeoutput > 0)&&(time > ((int)totaltime-1)*365.0)) {
	
	/*write to output file*/
	if ((fp = fopen("output.dat","ab")) == NULL) {
	  printf("could not open file for writing\n");
	  exit(1);
	}
	else {
	  for (i=0;i<N;i++) fwrite(&T[i],sizeof(double),1,fp);
	  for (i=0;i<Ne;i++) fwrite(&fracWAT[i],sizeof(double),1,fp);
	}
	fclose(fp);
	
      }/*if*/
      

      /*write to output file*/
      /*
	if ((fp = fopen("timetemp.dat","at")) == NULL) {
	printf("could not open file for writing\n");
	exit(1);
	}
	else {
	fprintf(fp,"%4.4e %4.4e\n",time,Ts);
	}
	fclose(fp);
      */

    }/*tt*/

  }/*yy*/


  /*if output required*/
  if (makeoutput > 0) {
    
    /*write to output file*/
    if ((fp = fopen("output.dat","ab")) == NULL) {
      printf("could not open file for writing\n");
      exit(1);
    }
    else {
      for (i=0;i<N;i++) fwrite(&Cr[i],sizeof(double),1,fp);
    }
    fclose(fp);
    
  }


  /*integrate annual cracking*/
  (*Ci) = 0.0;
  for (i=0;i<Ne;i++) (*Ci) += Cr[i]*dz[i];

  (*Tr) = 0.0;
  for (i=0;i<Ne;i++) {
    /*dwv[i] -= fabs(dwvs[i]);
      if (dwv[i] < 0.0) dwv[i] = 0.0;*/
    /*if (dwv[i] < 2.0) dwv[i] = 0.0; */
    (*Tr) += 0.5*beta*dwv[i]*dz[i]*ze[i];
  }

  /*write to output file*/
  /*
  if ((fp = fopen("Trtest.dat","wb")) == NULL) {
    printf("could not open file for writing\n");
    exit(1);
  }
  else {
    fwrite(&Ne,sizeof(int),1,fp);
    for (i=0;i<Ne;i++) fwrite(&ze[i],sizeof(double),1,fp);
    for (i=0;i<Ne;i++) fwrite(&dwv[i],sizeof(double),1,fp);
    for (i=0;i<Ne;i++) fwrite(&dwvs[i],sizeof(double),1,fp);
  }
  fclose(fp);
  */
  printf("Ci = %2.4lf, Tr = %2.4lf\n",(*Ci),(*Tr));
  
}


int main(int argc,char** argv) {

  int i,j;
  int Nt; /*# timesteps*/
  double dz0;
  double dt; /*timestep*/
  double *Temp;
  FILE *fp;
  char file[200];

  int N = 201; /*# Nodes*/
  int Ne = N - 1; /*# Elements*/
  int modID = atoi(argv[1]); /*Input argument 1*/
  double dTa = atof(argv[2]); /*Input argument 2*/
  double D = 20.0; /*Depth of profile (m)*/
  double cpICE = 2050.0; /*Specific heat Ice (J/kgK), http://www.engineeringtoolbox.com*/
  double cpWAT = 4210.0; /*Specific heat water (J/kgK)*/
  double cpBR = 790.0; /*Specific heat bedrock (J/kgK)*/
  double conBR  = 3.0; /*conductivity bedrock(W/(m*K))*/
  double conWAT = 0.56; /*conductivity water(W/(m*K))*/
  double conICE = 2.14; /*conductivity ice(W/(m*K)), Mottaghy & Rath, 2006*/
  double denWAT = 1000.0; /*density water (kg/m^3)*/
  double denICE = 916.7; /*density ice (kg/m^3), Mottaghy & Rath, 2006*/
  double denBR = 2650.0; /*density bedrock (kg/m^3)*/
  double L = 333.6e3; /*Latent heat of fusion water (J/kg), Mottaghy & Rath 2006*/
  double w = 1.0; /*scaling of mixture btw ice and water in phase-change zone, Mottaghy & Rath, 2006*/
  double Tliq = 0.0; /*Liquidus-temperature (deg C)*/
  double Tsol = -1.0; /*Solidus-temperature (deg C)*/
  double T0 = -6.0; /*Initial temperature*/
  double qb = 0.05; /*heat flow*/
  double po_b = 0.02; /*porosity bedrock*/
  double po_s = 0.30; /*porosity sediment*/
  double Hs = 0.5; /*initial regolith thickness(m)*/
  double beta = 0.05; /*strain parameter*/
  double spd = 24.0*3600.0; /*seconds-per-day*/
  double time = 0.0;
  double totaltime = 3.0;

  double minHs = 0.0; /*minimum regolith thickness*/
  double maxHs = 6.0; /*maximum regolith thickness*/
  double minT0 = -15.0; /*minimum mean-annual-temperature*/
  double maxT0 = 15.0; /*maximum mean-annual-temperature*/
  int nHs = 90; /*80*/
  int nT0 = 90; /*120*/


  /*read temperature data*/
  sprintf(file,"Temp.input"); /*filename*/
  if ((fp = fopen(file,"rb")) == NULL) {
    printf("Cannot open file : Temp.input\n");
    exit(1);
  }
  fread(&Nt,sizeof(int),1,fp);
  fread(&dt,sizeof(double),1,fp);
  Temp = malloc(Nt*sizeof(double)); /*allocate memory*/
  for (i=0;i<Nt;i++) fread(&Temp[i],sizeof(double),1,fp);
  fclose(fp);

  /*allocate memory*/
  int *sed = (int *) malloc(Ne*sizeof(int));
  int *cnn = (int *) malloc(Ne*sizeof(int));
  double *z = (double *) malloc(N*sizeof(double));
  double *dz = (double *) malloc(Ne*sizeof(double));
  double *ze = (double *) malloc(Ne*sizeof(double));
  double *T = (double *) malloc(N*sizeof(double));
  double *Tt = (double *) malloc(N*sizeof(double));
  double *Te = (double *) malloc(Ne*sizeof(double));
  double *dTdz = (double *) malloc(Ne*sizeof(double));
  double *po = (double *) malloc(Ne*sizeof(double));
  double *con = (double *) malloc(Ne*sizeof(double));
  double *cv = (double *) malloc(Ne*sizeof(double));
  double *denBulk = (double *) malloc(Ne*sizeof(double));
  double *Cr = (double *) malloc(Ne*sizeof(double));
  double *dwv = (double *) malloc(Ne*sizeof(double));
  double *dwvs = (double *) malloc(Ne*sizeof(double));
  double *fracWAT = (double *) malloc(Ne*sizeof(double));
  double *dfracWAT = (double *) malloc(Ne*sizeof(double));
  double *T0v = (double *) malloc(nT0*sizeof(double));
  double *Hsv = (double *) malloc(nHs*sizeof(double));
  double **Tcoef = (double **) malloc(5*sizeof(double*)); for (i=0;i<5;i++) Tcoef[i] = (double*) malloc(N*sizeof(double));
  double **Ci = (double **) malloc(nHs*sizeof(double*)); for (i=0;i<nHs;i++) Ci[i] = (double*) malloc(nT0*sizeof(double));
  double **Tr = (double **) malloc(nHs*sizeof(double*)); for (i=0;i<nHs;i++) Tr[i] = (double*) malloc(nT0*sizeof(double));

  /*define temperature vector*/
  for (i=0;i<nT0;i++) T0v[i] = minT0 + (double)i/((double)(nT0-1))*(maxT0-minT0);
  /*T0v[0] = 0.0;*/

  /*define sediment vector*/
  for (i=0;i<nHs;i++) Hsv[i] = (double)i/((double)(nHs-1));
  for (i=0;i<nHs;i++) Hsv[i] = minHs + (maxHs-minHs)*pow(Hsv[i],3.0);
  /*Hsv[0] = 0.0;*/
  
  /*
  T0v[0] = 1.0;
  Hsv[0] = 20.0;
  */

  /*define depth array*/
  dz0 = 1.0/((double)Ne);
  z[0] = 0.0;
  for (i=1;i<N;i++) z[i] = z[i-1] + dz0;
  for (i=1;i<N;i++) z[i] = D*pow(z[i],3.0);
  for (i=0;i<Ne;i++) {
    dz[i] = z[i+1] - z[i];
    ze[i] = .5*(z[i]+z[i+1]);
  }


  /*loop sediment thicknesses*/
  for (i=0;i<nHs;i++) { 

    /*loop MATs*/
    for (j=0;j<nT0;j++) {

      printf("Model %d/%d ",i*nT0+j+1,nHs*nT0);

      /*perform annual integration*/
      annual_integration(N,Ne,Nt,Temp,conBR,conICE,conWAT,denBR,denICE,denWAT,cpBR,cpICE,cpWAT,L,w,Tliq,Tsol,dTa,qb,T0v[j],po_s,po_b,beta,totaltime,dt,Hsv[i],sed,z,ze,dz,po,T,Tt,Te,dTdz,Tcoef,Cr,dwv,dwvs,fracWAT,dfracWAT,con,cv,denBulk,&Tr[i][j],&Ci[i][j],cnn,0);


    }/*j*/ 

  }/*i*/
    
  /*create output file*/
  sprintf(file,"frost_res_%d.dat",modID);
  if ((fp = fopen(file,"wb")) == NULL) {
    printf("could not open file for writing\n");
    exit(1);
  }
  else {
    fwrite(&nHs,sizeof(int),1,fp);
    fwrite(&nT0,sizeof(int),1,fp);
    for (i=0;i<nHs;i++) fwrite(&Hsv[i],sizeof(double),1,fp);
    for (i=0;i<nT0;i++) fwrite(&T0v[i],sizeof(double),1,fp);
    for (j=0;j<nT0;j++) for (i=0;i<nHs;i++) fwrite(&Ci[i][j],sizeof(double),1,fp);
    for (j=0;j<nT0;j++) for (i=0;i<nHs;i++) fwrite(&Tr[i][j],sizeof(double),1,fp);
  }
  fclose(fp);

  /*free memory*/
  free(sed);
  free(z);
  free(dz);
  free(ze);
  free(T);
  free(Te);
  free(dTdz);
  free(po);
  free(Cr);
  free(dwv);
  free(dwvs);
  free(T0v);
  free(Hsv);
  free(Ci);
  free(Tr);
  free(Tcoef);  
  free(Temp);

  exit(1);
}
/*References:
Mottaghy, D and Rath, V. Latent heat effects in subsurface heat transport modelling and their impact on paleotemperature reconstuctions. Journal of international Geophysics 164, p. 235-245. 2006*/
